<?php
/**
 * MainWP Jetpack Protect Plugin
 *
 * This file handles Plugin interactions.
 *
 * @package MainWP/Extensions
 */

namespace MainWP\Extensions\JetpackProtect;

/**
 * Class MainWP_Jetpack_Protect_Plugin
 */
class MainWP_Jetpack_Protect_Plugin {


	/**
	 * Protected static variable to hold the single instance of the class.
	 *
	 * @var mixed Default null
	 */
	private static $instance = null;

	/**
	 * Method get_instance()
	 *
	 * Create a new public static instance of
	 * MainWP_Jetpack_Protect_Plugin().
	 *
	 * @return void $instance New public static Instance.
	 */
	static function get_instance() {
		if ( null == self::$instance ) {
			self::$instance = new MainWP_Jetpack_Protect_Plugin();
		}
		return self::$instance;
	}

	/**
	 * MainWP_Jetpack_Protect_Plugin class constructor.
	 *
	 * @return void
	 */
	public function __construct() {

		add_action( 'admin_init', array( &$this, 'admin_init' ) );
	}

	/**
	 * Initiates the admin page when loaded & sets initial Class Variables.
	 */
	public function admin_init() {
		do_action( 'mainwp_ajax_add_action', 'mainwp_jetpack_protect_active_plugin', array( $this, 'ajax_active_plugin' ) );
		do_action( 'mainwp_ajax_add_action', 'mainwp_jetpack_protect_upgrade_plugin', array( $this, 'ajax_upgrade_plugin' ) );
		do_action( 'mainwp_ajax_add_action', 'mainwp_jetpack_protect_showhide_plugin', array( $this, 'ajax_showhide_plugin' ) );
		add_filter( 'mainwp_header_actions_right', array( $this, 'screen_options' ), 10, 2 );
		$this->handle_sites_screen_settings();
	}

	/**
	 * Method screen_options()
	 *
	 * Create Page Settings button.
	 *
	 * @param mixed $input Screen options button HTML.
	 *
	 * @return mixed Screen sptions button.
	 */
	public function screen_options( $input ) {
		if ( isset( $_GET['page'] ) && 'Extensions-Mainwp-Jetpack-Protect-Extension' == $_GET['page'] ) {
			if ( isset( $_GET['tab'] ) && 'dashboard' == $_GET['tab'] ) {
						$input .= '<a class="ui button basic icon" onclick="mainwp_jetpack_protect_screen_sites_options(); return false;" data-inverted="" data-position="bottom right" href="#" target="_blank" data-tooltip="' . esc_html__( 'Page Settings', 'mainwp-jetpack-protect-extension' ) . '">
					<i class="cog icon"></i>
				</a>';
			}
		}
		return $input;
	}

	public function gen_dashboard_table( $filter_groups = false ) {
		?>
		<table id="mainwp-jetpack-protect-sites-table" class="ui single line table" style="width:100%">
			<thead>
			<?php $this->gen_table_header_footer(); ?>
			</thead>
			<tbody>
				<?php

				$filter_site_ids       = null;
				$filter_group_site_ids = false;

				$group_empty = false;

				if ( false !== $filter_groups ) {
					$filter_group_site_ids = $this->get_site_ids_group_filter( $filter_groups );
				}

				if ( false === $filter_group_site_ids ) {
					$filter_site_ids = false;
				} elseif ( is_array( $filter_group_site_ids ) && empty( $filter_group_site_ids ) ) {
					$group_empty = true;
				} else {
					$filter_site_ids = $filter_group_site_ids;
				}

				$dbwebsites = false;

				if ( ! $group_empty ) {
					$sql_sites = MainWP_Jetpack_Protect_DB::get_instance()->get_sql_websites_ext_data();

					if ( ! empty( $sql_sites ) ) {
						$dbwebsites = MainWP_Jetpack_Protect_DB::get_instance()->query( $sql_sites );
					}
				}

				if ( ! $group_empty && $dbwebsites ) {
					while ( $dbwebsites && ( $website = MainWP_Jetpack_Protect_DB::fetch_object( $dbwebsites ) ) ) {

						if ( is_array( $filter_site_ids ) && ! in_array( $website->id, $filter_site_ids ) ) {
							continue;
						}

						$plugins = json_decode( $website->plugins, 1 );
						if ( is_array( $plugins ) && 0 != count( $plugins ) ) {
							foreach ( $plugins as $plugin ) {
								if ( 'jetpack-protect/jetpack-protect.php' == $plugin['slug'] ) {
									$site_data = MainWP_Jetpack_Protect_Utility::map_fields( $website, array( 'id', 'name', 'url' ) );

									$site_data['plugin_activated'] = isset( $plugin['active'] ) && $plugin['active'] ? 1 : 0;
									$site_data['plugin_version']   = $plugin['version'];

									// get upgrade info
									$plugin_upgrades = json_decode( $website->plugin_upgrades, 1 );
									if ( is_array( $plugin_upgrades ) && count( $plugin_upgrades ) > 0 ) {
										if ( isset( $plugin_upgrades['jetpack-protect/jetpack-protect.php'] ) ) {
											$upgrade = $plugin_upgrades['jetpack-protect/jetpack-protect.php'];
											if ( isset( $upgrade['update'] ) ) {
												$site_data['plugin_upgrade'] = array(
													'new_version' => isset( $upgrade['update']['new_version'] ) ? $upgrade['update']['new_version'] : '',
												);
											}
										}
									}

									$protect_data = MainWP_Jetpack_Protect_Utility::map_fields( $website, array( 'lasttime_scan', 'hide_plugin', 'lasttime_scan', 'connected', 'is_scan_active', 'num_vulnerabilities', 'num_core_vulnerabilities', 'num_plugins_vulnerabilities', 'num_themes_vulnerabilities', 'num_files_vulnerabilities', 'num_database_vulnerabilities' ) );

									$this->gen_dashboard_table_row( $site_data, $protect_data );

									break;
								}
							}
						}
					}
				}
				?>
			</tbody>
			<tfoot>
				<?php $this->gen_table_header_footer( false ); ?>
			</tfoot>
		</table>

		<script type="text/javascript">
			jQuery(function () {
				var $tbl_id = '#mainwp-jetpack-protect-sites-table';
				$jetpack_protect_sites_table =  jQuery( $tbl_id ).DataTable( {
					"stateSave": true,
					"stateDuration": 0, // forever
					"scrollX": true,
					"colReorder" : { columns: ":not(.check-column):not(:last-child)" },
					"lengthMenu": [ [10, 25, 50, 100, -1], [10, 25, 50, 100, "All"] ],
					"columnDefs": [ { "targets": 'no-sort', "orderable": false } ],
					"order": [ [ 1, "asc" ] ],
					"language": { "emptyTable": "No websites were found with the Jetpack Protect plugin installed." },
					"drawCallback": function( settings ) {
						jQuery( $tbl_id + ' .ui.checkbox').checkbox();
						jQuery( $tbl_id + ' .ui.dropdown').dropdown();
					},
					'select': {
						items: 'row',
						style: 'multi+shift',
						selector: 'tr>td:not(.not-selectable)'
					}
				} ).on('select', function (e, dt, type, indexes) {
					if( 'row' == type ){
						dt.rows(indexes)
						.nodes()
						.to$().find('td.check-column .ui.checkbox' ).checkbox('set checked');
					}
				}).on('deselect', function (e, dt, type, indexes) {
					if( 'row' == type ){
						dt.rows(indexes)
						.nodes()
						.to$().find('td.check-column .ui.checkbox' ).checkbox('set unchecked');
					}
				}).on( 'columns-reordered', function ( e, settings, details ) {
					console.log('columns-reordered');
					setTimeout(() => {
						jQuery( $tbl_id ).dropdown();
						jQuery( $tbl_id ).checkbox();
						mainwp_datatable_fix_menu_overflow($tbl_id );
					}, 1000);
				} );
				jQuery( $tbl_id + ' .ui.checkbox').checkbox();
				jQuery( $tbl_id + ' .ui.dropdown').dropdown();
				mainwp_datatable_fix_menu_overflow();
				_init_jetpack_protect_sites_screen();
			} );

			_init_jetpack_protect_sites_screen = function() {
				jQuery( '#mainwp-jetpack-protect-sites-screen-options-modal input[type=checkbox][id^="mainwp_show_column_"]' ).each( function() {
					var check_id = jQuery( this ).attr( 'id' );
					col_id = check_id.replace( "mainwp_show_column_", "" );
					try {
						$jetpack_protect_sites_table.column( '#' + col_id ).visible( jQuery(this).is( ':checked' ) );
					} catch(err) {
						// to fix js error.
					}
				} );
			};

			mainwp_jetpack_protect_screen_sites_options = function () {
				jQuery( '#mainwp-jetpack-protect-sites-screen-options-modal' ).modal( {
					allowMultiple: true,
					onHide: function () {
					}
				} ).modal( 'show' );

				jQuery( '#jetpack-protect-sites-screen-options-form' ).submit( function() {
					if ( jQuery('input[name=reset_jetpack_protect_sites_columns_order]').attr('value') == 1 ) {
						$jetpack_protect_sites_table.colReorder.reset();
					}
					jQuery( '#mainwp-jetpack-protect-sites-screen-options-modal' ).modal( 'hide' );
				} );
				return false;
			};


		</script>
		<?php
		$this->render_screen_options();
	}

	public function gen_dashboard_widget_table() {
		$dbwebsites = false;
		$sql_sites  = MainWP_Jetpack_Protect_DB::get_instance()->get_sql_websites_ext_data();

		if ( ! empty( $sql_sites ) ) {
			$dbwebsites = MainWP_Jetpack_Protect_DB::get_instance()->query( $sql_sites );
		}

		?>
		<table class="ui table" id="mainwp-jetpack-protect-widget-table">
			<thead>
				<tr>
					<th><?php esc_html_e( 'Site', 'mainwp-jetpack-protect-extension' ); ?></th>
					<th class="center aligned"><span data-inverted="" data-position="left center" data-tooltip="<?php esc_attr_e( 'WordPress core threats', 'mainwp-jetpack-protect-extension' ); ?>"><i class="wordpress icon"></i></span></th>  <?php //phpcs:ignore -- to fix WordPress word. ?>
					<th class="center aligned"><span data-inverted="" data-position="left center" data-tooltip="<?php esc_attr_e( 'Plugins threats', 'mainwp-jetpack-protect-extension' ); ?>"><i class="plug icon"></i></span></th>
					<th class="center aligned"><span data-inverted="" data-position="left center" data-tooltip="<?php esc_attr_e( 'Themes threats', 'mainwp-jetpack-protect-extension' ); ?>"><i class="tint icon"></i></span></th>
				</tr>
			</thead>
			<tbody>
				<?php
				while ( $dbwebsites && ( $website = MainWP_Jetpack_Protect_DB::fetch_object( $dbwebsites ) ) ) {

					$plugins = json_decode( $website->plugins, 1 );
					if ( is_array( $plugins ) && 0 != count( $plugins ) ) {
						foreach ( $plugins as $plugin ) {
							if ( 'jetpack-protect/jetpack-protect.php' == $plugin['slug'] ) {

								$protect_data = MainWP_Jetpack_Protect_Utility::map_fields( $website, array( 'lasttime_scan', 'hide_plugin', 'scan_time', 'connected', 'is_scan_active', 'num_vulnerabilities', 'num_plugins_vulnerabilities', 'num_themes_vulnerabilities', 'num_core_vulnerabilities', 'num_files_vulnerabilities', 'num_database_vulnerabilities' ) );

								$this->gen_widget_table_row( $website, $protect_data );
								break;
							}
						}
					}
				}
				?>
			</tbody>
		</table>
		<script type="text/javascript">
		jQuery( document ).ready( function () {
			jQuery( '#mainwp-jetpack-protect-widget-table' ).DataTable( {
				"lengthMenu": [ [10, 25, 50, 100, -1], [10, 25, 50, 100, "All"] ],
			} );
		} );
		</script>
		<?php
	}

	/**
	 * Get site table columns.
	 *
	 * @return array colunns.
	 */
	public function get_columns() {
		return array(
			'site'            => esc_html__( 'Site', 'mainwp-jetpack-protect-extension' ),
			'sign-in'         => '<i class="sign in icon"></i>',
			'url'             => esc_html__( 'URL', 'mainwp-jetpack-protect-extension' ),
			'wp-core'         => esc_html__( 'WP', 'mainwp-jetpack-protect-extension' ),
			'plugins'         => esc_html__( 'Plugins', 'mainwp-jetpack-protect-extension' ),
			'themes'          => esc_html__( 'Themes', 'mainwp-jetpack-protect-extension' ),
			'files'           => esc_html__( 'Files', 'mainwp-jetpack-protect-extension' ),
			'database'        => esc_html__( 'Database', 'mainwp-jetpack-protect-extension' ),
			'last_check_date' => esc_html__( 'Last check date', 'mainwp-jetpack-protect-extension' ),
			'connected'       => esc_html__( 'Connected', 'mainwp-jetpack-protect-extension' ),
			'version'         => esc_html__( 'Version', 'mainwp-jetpack-protect-extension' ),
			'hide'            => esc_html__( 'Hide', 'mainwp-jetpack-protect-extension' ),
		);
	}

	/**
	 * Get table header/footer.
	 */
	public function gen_table_header_footer( $top = true ) {
		?>
		<tr>
			<th class="no-sort collapsing check-column" id="check-column" ><span class="ui checkbox"><input type="checkbox" id="<?php echo $top ? 'cb-select-all-top' : 'cb-select-all-bottom'; ?>"></span></th>
			<th id="site"><?php esc_html_e( 'Site', 'mainwp-jetpack-protect-extension' ); ?></th>
			<th class="no-sort collapsing" id="sign-in"><i class="sign in icon"></i></th>
			<th id="url"><?php esc_html_e( 'URL', 'mainwp-jetpack-protect-extension' ); ?></th>
			<th id="wp-core" class="center aligned collapsing"><span data-inverted="" data-position="left center" data-tooltip="<?php esc_attr_e( 'WordPress core threats', 'mainwp-jetpack-protect-extension' ); ?>"><i class="wordpress icon"></i></span></th> <?php //phpcs:ignore -- to fix WordPress word. ?>
			<th id="plugins" class="center aligned collapsing"><span data-inverted="" data-position="left center" data-tooltip="<?php esc_attr_e( 'Plugins threats', 'mainwp-jetpack-protect-extension' ); ?>"><i class="plug icon"></i></span></th>
			<th id="themes" class="center aligned collapsing"><span data-inverted="" data-position="left center" data-tooltip="<?php esc_attr_e( 'Themes threats', 'mainwp-jetpack-protect-extension' ); ?>"><i class="tint icon"></i></span></th>
			<th id="files" class="center aligned collapsing"><span data-inverted="" data-position="left center" data-tooltip="<?php esc_attr_e( 'Files threats', 'mainwp-jetpack-protect-extension' ); ?>"><i class="code icon"></i></span></th>
			<th id="database" class="center aligned collapsing"><span data-inverted="" data-position="left center" data-tooltip="<?php esc_attr_e( 'Database threats', 'mainwp-jetpack-protect-extension' ); ?>"><i class="border all icon"></i></span></th>
			<th id="last_check_date" class="center aligned collapsing"><?php esc_html_e( 'Last check date', 'mainwp-jetpack-protect-extension' ); ?></th>
			<th id="connected" class="center aligned collapsing"><?php esc_html_e( 'Jetpack Connection', 'mainwp-jetpack-protect-extension' ); ?></th>
			<th id="version"><?php esc_html_e( 'Version', 'mainwp-jetpack-protect-extension' ); ?></th>
			<th id="hide"><i class="eye slash outline icon"></i></th>
			<th class="no-sort collapsing" id="actions"></th>
		</tr>
		<?php
	}

	/**
	 * Get table row.
	 */
	public function gen_dashboard_table_row( $website, $protect_data ) {

		$plugin_slug = 'jetpack-protect/jetpack-protect.php';
		$plugin_name = 'Jetpack Protect';

		$website_id = intval( $website['id'] );

		$class_active = ( isset( $website['plugin_activated'] ) && ! empty( $website['plugin_activated'] ) ) ? '' : 'negative';
		$class_update = ( isset( $website['plugin_upgrade'] ) ) ? 'warning' : '';
		$class_update = ( 'negative' == $class_active ) ? 'negative' : $class_update;

		$location = 'admin.php?page=jetpack-protect';

		$num_vuln = isset( $protect_data['num_vulnerabilities'] ) ? intval( $protect_data['num_vulnerabilities'] ) : false;

		$num_plugins_vuln  = isset( $protect_data['num_plugins_vulnerabilities'] ) ? intval( $protect_data['num_plugins_vulnerabilities'] ) : false;
		$num_themes_vuln   = isset( $protect_data['num_themes_vulnerabilities'] ) ? intval( $protect_data['num_themes_vulnerabilities'] ) : false;
		$num_files_vuln    = isset( $protect_data['num_files_vulnerabilities'] ) ? intval( $protect_data['num_files_vulnerabilities'] ) : false;
		$num_database_vuln = isset( $protect_data['num_database_vulnerabilities'] ) ? intval( $protect_data['num_database_vulnerabilities'] ) : false;
		$num_core_vuln     = isset( $protect_data['num_core_vulnerabilities'] ) ? intval( $protect_data['num_core_vulnerabilities'] ) : false;

		$lasttime_scan = isset( $protect_data['lasttime_scan'] ) ? intval( $protect_data['lasttime_scan'] ) : false;
		$scan_actived  = isset( $protect_data['is_scan_active'] ) ? intval( $protect_data['is_scan_active'] ) : false;

		$v_color   = MainWP_Jetpack_Protect_Utility::get_vulnerabilities_number_color( $num_core_vuln );
		$vp_color  = MainWP_Jetpack_Protect_Utility::get_vulnerabilities_number_color( $num_plugins_vuln );
		$vth_color = MainWP_Jetpack_Protect_Utility::get_vulnerabilities_number_color( $num_themes_vuln );
		$vf_color  = MainWP_Jetpack_Protect_Utility::get_vulnerabilities_number_color( $num_files_vuln );
		$vd_color  = MainWP_Jetpack_Protect_Utility::get_vulnerabilities_number_color( $num_database_vuln );

		$is_connected = isset( $protect_data['connected'] ) && ! empty( $protect_data['connected'] ) ? true : false;

		$disabled_tooltip = '';
		if ( ! $scan_actived ) {
			$disabled_tooltip = esc_html__( 'Jetpack Scan required.', 'mainwp-jetpack-protect-extension' );
			$vf_color         = 'greyed';
			$vd_color         = 'greyed';
		}
		?>
		<tr class="<?php echo $class_active . ' ' . $class_update; ?>" website-id="<?php echo $website_id; ?>" plugin-name="<?php echo $plugin_name; ?>" plugin-slug="<?php echo $plugin_slug; ?>" version="<?php echo ( isset( $website['plugin_version'] ) ) ? $website['plugin_version'] : 'N/A'; ?>">
			<td class="check-column"><span class="ui checkbox"><input type="checkbox" name="checked[]"></span></td>
			<td class="website-name"><a href="admin.php?page=managesites&dashboard=<?php echo $website_id; ?>" data-tooltip="<?php esc_attr_e( 'Click to jump to the site Overview page', 'mainwp-jetpack-protect-extension' ); ?>" data-position="right center" data-inverted=""><?php echo stripslashes( $website['name'] ); ?></a></td>
			<td><a target="_blank" href="admin.php?page=SiteOpen&newWindow=yes&websiteid=<?php echo $website_id; ?>&_opennonce=<?php echo wp_create_nonce( 'mainwp-admin-nonce' ); ?>" data-tooltip="<?php esc_attr_e( 'Jump to the site WP Admin', 'mainwp-jetpack-protect-extension' ); ?>" data-position="right center" data-inverted=""><i class="sign in icon"></i></a></td>
			<td><a href="<?php echo $website['url']; ?>" target="_blank" data-tooltip="<?php esc_attr_e( 'Jump to the site front page', 'mainwp-jetpack-protect-extension' ); ?>" data-position="right center" data-inverted=""><?php echo $website['url']; ?></a></td>
			<?php if ( ! empty( $lasttime_scan ) ) : ?>
			<td class="center aligned">
				<?php if ( false !== $num_core_vuln ) : ?>
						<span data-tooltip="<?php esc_attr_e( 'Number of WP vulnerabilities. Click to see details.', 'mainwp-jetpack-protect-extension' ); ?>" data-position="left center" data-inverted="">
							<a class="ui large basic label <?php echo $v_color; ?>" href="admin.php?page=ManageSitesJetpackProtect&id=<?php echo $website_id; ?>">
								<?php echo $num_core_vuln; ?>
							</a>
						</span>
				<?php else : ?>
					<span data-tooltip="<?php esc_attr_e( 'No data available for the site.', 'mainwp-jetpack-protect-extension' ); ?>" data-position="left center" data-inverted=""><i class="exclamation circle icon"></i></span>
				<?php endif; ?>
			</td>
			<td class="center aligned">
				<?php if ( false !== $num_plugins_vuln ) : ?>
						<span data-tooltip="<?php esc_attr_e( 'Number of Plugins vulnerabilities. Click to see details.', 'mainwp-jetpack-protect-extension' ); ?>" data-position="left center" data-inverted="">
							<a class="ui large basic label <?php echo $vp_color; ?>" href="admin.php?page=ManageSitesJetpackProtect&id=<?php echo $website_id; ?>">
								<?php echo $num_plugins_vuln; ?>
							</a>
						</span>
				<?php else : ?>
					<span data-tooltip="<?php esc_attr_e( 'No data available for the site.', 'mainwp-jetpack-protect-extension' ); ?>" data-position="left center" data-inverted=""><i class="exclamation circle icon"></i></span>
				<?php endif; ?>
			</td>
			<td class="center aligned">
				<?php if ( false !== $num_themes_vuln ) : ?>
						<span data-tooltip="<?php esc_attr_e( 'Number of Themes vulnerabilities. Click to see details.', 'mainwp-jetpack-protect-extension' ); ?>" data-position="left center" data-inverted="">
							<a class="ui large basic label <?php echo $vth_color; ?>" href="admin.php?page=ManageSitesJetpackProtect&id=<?php echo $website_id; ?>">
								<?php echo $num_themes_vuln; ?>
							</a>
						</span>
				<?php else : ?>
					<span data-tooltip="<?php esc_attr_e( 'No data available for the site.', 'mainwp-jetpack-protect-extension' ); ?>" data-position="left center" data-inverted=""><i class="exclamation circle icon"></i></span>
				<?php endif; ?>
			</td>

			<td class="center aligned">
				<?php
				if ( false !== $num_files_vuln ) :
					if ( $scan_actived ) {
						$disabled_tooltip = esc_html__( 'Number of Files vulnerabilities. Click to see details.', 'mainwp-jetpack-protect-extension' );
					}
					?>
						<span data-tooltip="<?php echo $disabled_tooltip; ?>" data-position="left center" data-inverted="">
							<a class="ui large basic label <?php echo $vf_color; ?>" href="admin.php?page=ManageSitesJetpackProtect&id=<?php echo $website_id; ?>">
								<?php echo $num_files_vuln; ?>
							</a>
						</span>
				<?php else : ?>
					<span data-tooltip="<?php esc_attr_e( 'No data available for the site.', 'mainwp-jetpack-protect-extension' ); ?>" data-position="left center" data-inverted=""><i class="exclamation circle icon"></i></span>
				<?php endif; ?>
			</td>

			<td class="center aligned">
				<?php
				if ( false !== $num_database_vuln ) :
					if ( $scan_actived ) {
						$disabled_tooltip = esc_html__( 'Number of Database vulnerabilities. Click to see details.', 'mainwp-jetpack-protect-extension' );
					}
					?>
						<span data-tooltip="<?php echo $disabled_tooltip; ?>" data-position="left center" data-inverted="">
							<a class="ui large basic label <?php echo $vd_color; ?>" href="admin.php?page=ManageSitesJetpackProtect&id=<?php echo $website_id; ?>">
								<?php echo $num_database_vuln; ?>
							</a>
						</span>
				<?php else : ?>
					<span data-tooltip="<?php esc_attr_e( 'No data available for the site.', 'mainwp-jetpack-protect-extension' ); ?>" data-position="left center" data-inverted=""><i class="exclamation circle icon"></i></span>
				<?php endif; ?>
			</td>
			<?php else : ?>
				<td class="center aligned"></td>
				<td class="center aligned"></td>
				<td class="center aligned"></td>
				<td class="center aligned"></td>
				<td class="center aligned"></td>
			<?php endif; ?>
			<td class="center aligned">
				<?php if ( ! empty( $lasttime_scan ) ) : ?>
					<?php echo MainWP_Jetpack_Protect_Utility::format_timestamp( MainWP_Jetpack_Protect_Utility::get_timestamp( $lasttime_scan ) ); ?>
				<?php else : ?>
					<span data-tooltip="<?php esc_attr_e( 'No data available for the site.', 'mainwp-jetpack-protect-extension' ); ?>" data-position="left center" data-inverted=""><i class="exclamation circle icon"></i></span>
				<?php endif; ?>
			</td>
			<td  class="center aligned"><span class="set-connection"></span>
				<?php if ( isset( $protect_data['connected'] ) && ! empty( $protect_data['connected'] ) ) : ?>
					<span data-tooltip="<?php esc_html_e( 'Connected', 'mainwp-jetpack-protect-extension' ); ?>" data-inverted="" data-position="left center"><a href="javascript:void(0)" class="mainwp-jetpack-protect-connect-disconnect" current-status="<?php echo $is_connected ? 'connect' : 'disconnect'; ?>"><i class="icons"><i class="cloud icon grey big" style="color:#eee!important"></i><i class="corner large green check icon" style="color:#7fb100!important"></i></i></a><span>
				<?php else : ?>
					<span data-tooltip="<?php esc_html_e( 'Disconnected', 'mainwp-jetpack-protect-extension' ); ?>" data-inverted="" data-position="left center"><a href="javascript:void(0)" class="mainwp-jetpack-protect-connect-disconnect" current-status="<?php echo $is_connected ? 'connect' : 'disconnect'; ?>"><i class="icons"><i class="cloud icon grey big" style="color:#eee!important"></i><i class="corner large red times icon" style="color:#ad0000!important"></i></i></a><span>
				<?php endif; ?>
			</td>
			<td><span class="updating"></span> <?php echo ( isset( $website['plugin_upgrade'] ) ) ? '<i class="exclamation circle icon"></i>' : ''; ?> <?php echo ( isset( $website['plugin_version'] ) ) ? $website['plugin_version'] : 'N/A'; ?></td>
			<td><span class="visibility"></span> <span class="wp-reports-visibility"><?php echo ( 1 == $protect_data['hide_plugin'] ) ? esc_html__( 'Yes', 'mainwp-jetpack-protect-extension' ) : esc_html__( 'No', 'mainwp-jetpack-protect-extension' ); ?></span></td>
			<td class="not-selectable">
				<div class="ui right pointing dropdown icon mini basic green button" style="z-index:999">
					<a href="javascript:void(0);" data-tooltip="<?php esc_attr_e( 'See more options', 'mainwp-jetpack-protect-extension' ); ?>" data-position="left center" data-inverted=""><i class="ellipsis horizontal icon"></i></a>
					<div class="menu">
						<a class="item" href="admin.php?page=managesites&dashboard=<?php echo $website_id; ?>"><?php esc_html_e( 'Overview', 'mainwp-jetpack-protect-extension' ); ?></a>
						<a class="item" href="admin.php?page=managesites&id=<?php echo $website_id; ?>"><?php esc_html_e( 'Edit', 'mainwp-jetpack-protect-extension' ); ?></a>

						<a class="item" href="admin.php?page=SiteOpen&newWindow=yes&websiteid=<?php echo $website_id; ?>&location=<?php echo base64_encode( $location ); ?>&_opennonce=<?php echo wp_create_nonce( 'mainwp-admin-nonce' ); ?>" target="_blank"><?php esc_html_e( 'Go to Jetpack Protect', 'mainwp-jetpack-protect-extension' ); ?></a>
							<?php if ( 1 == $protect_data['hide_plugin'] ) : ?>
							<a class="item mainwp-jetpack-protect-showhide-plugin" href="javascript:void(0);" showhide="show"><?php esc_html_e( 'Unhide Jetpack Protect', 'mainwp-jetpack-protect-extension' ); ?></a>
							<?php else : ?>
							<a class="item mainwp-jetpack-protect-showhide-plugin" href="javascript:void(0);" showhide="hide"><?php esc_html_e( 'Hide Jetpack Protect', 'mainwp-jetpack-protect-extension' ); ?></a>
							<?php endif; ?>
							<?php if ( isset( $website['plugin_activated'] ) && empty( $website['plugin_activated'] ) ) : ?>
							<a class="item mainwp-jetpack-protect-activate-plugin" href="javascript:void(0);"><?php esc_html_e( 'Activate Jetpack Protect', 'mainwp-jetpack-protect-extension' ); ?></a>
							<?php endif; ?>
							<?php if ( isset( $website['plugin_upgrade'] ) ) : ?>
							<a class="item mainwp-jetpack-protect-update-plugin" href="javascript:void(0);"><?php esc_html_e( 'Update Jetpack Protect', 'mainwp-jetpack-protect-extension' ); ?></a>
							<?php endif; ?>
						<a class="item mainwp-jetpack-protect-connect-disconnect" href="javascript:void(0)" current-status="<?php echo $is_connected ? 'connect' : 'disconnect'; ?>"><?php echo $is_connected ? esc_html__( 'Disconnect Jetpack Protect', 'mainwp-jetpack-protect-extension' ) : esc_html__( 'Connect Jetpack Protect', 'mainwp-jetpack-protect-extension' ); ?></a>
					</div>
				</div>
			</td>
		</tr>
		<?php
	}

	/**
	 * Get table row.
	 */
	public function gen_widget_table_row( $website, $protect_data ) {

		$website_id = intval( $website->id );

		$num_vuln         = isset( $protect_data['num_vulnerabilities'] ) ? intval( $protect_data['num_vulnerabilities'] ) : false;
		$num_plugins_vuln = isset( $protect_data['num_plugins_vulnerabilities'] ) ? intval( $protect_data['num_plugins_vulnerabilities'] ) : false;
		$num_themes_vuln  = isset( $protect_data['num_themes_vulnerabilities'] ) ? intval( $protect_data['num_themes_vulnerabilities'] ) : false;
		$num_core_vuln    = isset( $protect_data['num_core_vulnerabilities'] ) ? intval( $protect_data['num_core_vulnerabilities'] ) : false;

		$v_color   = MainWP_Jetpack_Protect_Utility::get_vulnerabilities_number_color( $num_core_vuln );
		$vp_color  = MainWP_Jetpack_Protect_Utility::get_vulnerabilities_number_color( $num_plugins_vuln );
		$vth_color = MainWP_Jetpack_Protect_Utility::get_vulnerabilities_number_color( $num_themes_vuln );

		?>

		<?php if ( 0 < $num_vuln ) : ?>
		<tr>
			<td>
				<a href="admin.php?page=ManageSitesJetpackProtect&id=<?php echo $website_id; ?>" data-tooltip="<?php esc_attr_e( 'Click to jump to the site Jetpack Protect overview page', 'mainwp-jetpack-protect-extension' ); ?>" data-position="right center" data-inverted=""><?php echo stripslashes( $website->name ); ?></a><br/>
				<div><span class="ui small text"><?php esc_html_e( 'Latest results as of: ', 'mainwp-jetpack-protect-extension' ); ?> <?php echo MainWP_Jetpack_Protect_Utility::format_timestamp( MainWP_Jetpack_Protect_Utility::get_timestamp( $protect_data['lasttime_scan'] ) ); ?></span></div>
			</td>
			<td class="center aligned">
				<?php if ( false !== $num_core_vuln ) : ?>
				<span data-tooltip="<?php esc_attr_e( 'Number of WP core vulnerabilities. Click to see details.', 'mainwp-jetpack-protect-extension' ); ?>" data-position="left center" data-inverted=""><a class="ui mini compact basic button <?php echo $v_color; ?>" href="admin.php?page=ManageSitesJetpackProtect&id=<?php echo $website_id; ?>"><?php echo $num_core_vuln; ?></a></span>
				<?php else : ?>
					<?php echo 'N/A'; ?>
				<?php endif; ?>
			</td>
			<td class="center aligned">
				<?php if ( false !== $num_plugins_vuln ) : ?>
				<span data-tooltip="<?php esc_attr_e( 'Number of Plugins vulnerabilities. Click to see details.', 'mainwp-jetpack-protect-extension' ); ?>" data-position="left center" data-inverted=""><a class="ui mini compact basic button <?php echo $vp_color; ?>" href="admin.php?page=ManageSitesJetpackProtect&id=<?php echo $website_id; ?>"><?php echo $num_plugins_vuln; ?></a></span>
				<?php else : ?>
					<?php echo 'N/A'; ?>
				<?php endif; ?>
			</td>
			<td class="center aligned">
				<?php if ( false !== $num_themes_vuln ) : ?>
				<span data-tooltip="<?php esc_attr_e( 'Number of Themes vulnerabilities. Click to see details.', 'mainwp-jetpack-protect-extension' ); ?>" data-position="left center" data-inverted=""><a class="ui mini compact basic button <?php echo $vth_color; ?>" href="admin.php?page=ManageSitesJetpackProtect&id=<?php echo $website_id; ?>"><?php echo $num_themes_vuln; ?></a></span>
				<?php else : ?>
					<?php echo 'N/A'; ?>
				<?php endif; ?>
			</td>
		</tr>
		<?php endif; ?>
		<?php
	}

	public static function get_site_ids_group_filter( $groups_ids ) {

		global $mainJetpackProtectExtensionActivator;

		if ( ! empty( $groups_ids ) ) {
			$group_websites = apply_filters( 'mainwp_getdbsites', $mainJetpackProtectExtensionActivator->get_child_file(), $mainJetpackProtectExtensionActivator->get_child_key(), array(), $groups_ids );
			$sites          = array();
			foreach ( $group_websites as $site ) {
				$sites[] = $site->id;
			}
			return $sites;
		}

		return false;
	}

	public function gen_actions_bar( $filter_groups ) {

		global $mainJetpackProtectExtensionActivator;
		$groups = apply_filters( 'mainwp_getgroups', $mainJetpackProtectExtensionActivator->get_child_file(), $mainJetpackProtectExtensionActivator->get_child_key(), null );

		if ( ! is_array( $filter_groups ) ) {
			$filter_groups = array();
		}
		?>
		<div class="mainwp-actions-bar">
			<div class="ui two columns grid">
				<div class="ui mini form column">
					<select id="mainwp-jetpack-protect-actions" class="ui mini dropdown">
						<option selected="selected" value=""><?php esc_html_e( 'Bulk Actions', 'mainwp-jetpack-protect-extension' ); ?></option>
						<option value="activate-selected"><?php esc_html_e( 'Activate Jetpack Protect' ); ?></option>
						<option value="update-selected"><?php esc_html_e( 'Update Jetpack Protect' ); ?></option>
						<option value="hide-selected"><?php esc_html_e( 'Hide Jetpack Protect' ); ?></option>
						<option value="show-selected"><?php esc_html_e( 'Unhide Jetpack Protect' ); ?></option>
						<option value="connect-jetpack"><?php esc_html_e( 'Connect Jetpack' ); ?></option>
						<option value="disconnect-jetpack"><?php esc_html_e( 'Disconnect Jetpack' ); ?></option>
					</select>
					<input type="button" value="<?php esc_html_e( 'Apply' ); ?>" class="ui mini button" id="mainwp-jetpack-protect-actions-button">
				</div>
				<div class="right aligned column">
					<form method="post" action="admin.php?page=Extensions-Mainwp-Jetpack-Protect-Extension&tab=dashboard" id="mainwp-jetpack-protect-groups-selection-form">
						<select name="mainwp-jetpack-protect-groups-selection" id="mainwp-jetpack-protect-groups-selection" multiple="" class="ui mini dropdown" >
							<option value=""><?php esc_html_e( 'All tags', 'mainwp-jetpack-protect-extension' ); ?></option>
							<?php
							if ( is_array( $groups ) && count( $groups ) > 0 ) {
								foreach ( $groups as $group ) {

									$_select = '';
									if ( in_array( $group['id'], $filter_groups ) ) {
										$_select = 'selected ';
									}

									echo '<option value="' . $group['id'] . '" ' . $_select . '>' . $group['name'] . '</option>';
								}
							}
							?>
						</select>
						<input class="ui mini button" type="button" name="mainwp-jetpack-protect-sites-filter-button" id="mainwp-jetpack-protect-sites-filter-button" value="<?php esc_html_e( 'Filter Sites', 'mainwp-jetpack-protect-extension' ); ?>">
					</form>
				</div>
			</div>
		</div>
		<?php
	}

	/**
	 * Method ajax_active_plugin()
	 *
	 * Ajax active plugin.
	 */
	public function ajax_active_plugin() {
		do_action( 'mainwp_secure_request', 'mainwp_jetpack_protect_active_plugin' );
		do_action( 'mainwp_action_activeplugin' );
		die();
	}

	/**
	 * Method ajax_upgrade_plugin()
	 *
	 * Ajax upgrade plugin.
	 */
	public function ajax_upgrade_plugin() {
		do_action( 'mainwp_secure_request', 'mainwp_jetpack_protect_upgrade_plugin' );
		do_action( 'mainwp_action_upgradeplugintheme' );
		die();
	}

	/**
	 * Method ajax_showhide_plugin()
	 *
	 * Ajax show/hide plugin.
	 */
	public function ajax_showhide_plugin() {
		do_action( 'mainwp_secure_request', 'mainwp_jetpack_protect_showhide_plugin' );
		$siteid   = isset( $_POST['websiteId'] ) ? intval( $_POST['websiteId'] ) : null;
		$showhide = isset( $_POST['showhide'] ) ? sanitize_text_field( wp_unslash( $_POST['showhide'] ) ) : null;
		if ( null !== $siteid && null !== $showhide ) {
			global $mainJetpackProtectExtensionActivator;
			$post_data   = array(
				'mwp_action' => 'set_showhide',
				'showhide'   => $showhide,
			);
			$information = apply_filters( 'mainwp_fetchurlauthed', $mainJetpackProtectExtensionActivator->get_child_file(), $mainJetpackProtectExtensionActivator->get_child_key(), $siteid, 'jetpack_protect', $post_data );

			if ( is_array( $information ) && isset( $information['result'] ) && 'SUCCESS' === $information['result'] ) {
				$update = array(
					'wpid'        => $siteid,
					'hide_plugin' => ( 'hide' === $showhide ) ? 1 : 0,
				);
				MainWP_Jetpack_Protect_DB::get_instance()->update_protect( $update );

			}
			die( json_encode( $information ) );
		}
		die();
	}


	/**
	 * Method handle_sites_screen_settings()
	 *
	 * Handle sites screen settings
	 */
	public function handle_sites_screen_settings() {
		if ( isset( $_POST['mainwp_jetpack_protect_sites_screen_options_nonce'] ) ) {

			$secure = apply_filters( 'mainwp_check_security_request', false, 'mainwp_jetpack_protect_sites_screen_options_nonce', 'mainwp_jetpack_protect_sites_screen_options_nonce' );
			if ( ! $secure ) {
				wp_die( 'Invalid request.' );
				return;
			}

			$show_cols = array();
			foreach ( array_map( 'sanitize_text_field', wp_unslash( $_POST ) ) as $key => $val ) {
				if ( false !== strpos( $key, 'mainwp_show_column_' ) ) {
					$col               = str_replace( 'mainwp_show_column_', '', $key );
					$show_cols[ $col ] = 1;
				}
			}
			if ( isset( $_POST['show_columns_name'] ) ) {
				foreach ( array_map( 'sanitize_text_field', wp_unslash( $_POST['show_columns_name'] ) ) as $col ) {
					if ( ! isset( $show_cols[ $col ] ) ) {
						$show_cols[ $col ] = 0; // uncheck, hide columns.
					}
				}
			}
			$user = wp_get_current_user();
			if ( $user ) {
				update_user_option( $user->ID, 'mainwp_jetpack_protect_settings_sites_show_columns', $show_cols, true );
			}
		}
	}


	/**
	 * Render screen options.
	 *
	 * @return void
	 */
	public function render_screen_options() {

		$columns = self::get_columns();

		$show_cols = get_user_option( 'mainwp_jetpack_protect_settings_sites_show_columns' );

		if ( ! is_array( $show_cols ) ) {
			$show_cols = array();
		}
		?>
		<div class="ui modal" id="mainwp-jetpack-protect-sites-screen-options-modal">
		<i class="close icon"></i>
			<div class="header"><?php esc_html_e( 'Page Settings', 'mainwp-jetpack-protect-extension' ); ?></div>
			<div class="scrolling content ui form">
				<form method="POST" action="" id="jetpack-protect-sites-screen-options-form" name="jetpack_protect_sites_screen_options_form">
					<?php wp_nonce_field( 'mainwp-admin-nonce' ); ?>
					<input type="hidden" name="mainwp_jetpack_protect_sites_screen_options_nonce" value="<?php echo wp_create_nonce( 'mainwp_jetpack_protect_sites_screen_options_nonce' ); ?>" />
						<div class="ui grid field">
							<label class="six wide column"><?php esc_html_e( 'Show columns', 'mainwp-jetpack-protect-extension' ); ?></label>
							<div class="ten wide column">
								<ul class="mainwp_hide_wpmenu_checkboxes">
									<?php
									foreach ( $columns as $name => $title ) {
										?>
										<li>
											<div class="ui checkbox">
												<input type="checkbox"
												<?php
												$show_col = ! isset( $show_cols[ $name ] ) || ( 1 == $show_cols[ $name ] );
												if ( $show_col ) {
													echo 'checked="checked"';
												}
												?>
												id="mainwp_show_column_<?php echo esc_attr( $name ); ?>" name="mainwp_show_column_<?php echo esc_attr( $name ); ?>" value="<?php echo esc_attr( $name ); ?>">
												<label for="mainwp_show_column_<?php echo esc_attr( $name ); ?>" ><?php echo $title; ?></label>
												<input type="hidden" value="<?php echo esc_attr( $name ); ?>" name="show_columns_name[]" />
											</div>
										</li>
										<?php
									}
									?>
								</ul>
							</div>
					</div>
				</div>
			<div class="actions">
					<div class="ui two columns grid">
						<div class="left aligned column">
							<span data-tooltip="<?php esc_attr_e( 'Returns this page to the state it was in when installed. The feature also restores any column you have moved through the drag and drop feature on the page.', 'mainwp-jetpack-protect-extension' ); ?>" data-inverted="" data-position="top center"><input type="button" class="ui button" name="reset" id="reset-jetpack-protect-sites-settings" value="<?php esc_attr_e( 'Reset Page', 'mainwp-jetpack-protect-extension' ); ?>" /></span>
						</div>
						<div class="ui right aligned column">
					<input type="submit" class="ui green button" name="btnSubmit" id="submit-jetpack-protect-sites-settings" value="<?php esc_attr_e( 'Save Settings', 'mainwp-jetpack-protect-extension' ); ?>" />
				</div>
					</div>
				</div>
				<input type="hidden" name="reset_jetpack_protect_sites_columns_order" value="0">
			</form>
		</div>
		<div class="ui small modal" id="mainwp-jetpack-protect-sites-site-preview-screen-options-modal">
		<i class="close icon"></i>
			<div class="header"><?php esc_html_e( 'Page Settings', 'mainwp-jetpack-protect-extension' ); ?></div>
			<div class="scrolling content ui form">
				<span><?php esc_html_e( 'Would you like to turn on home screen previews?  This function queries WordPress.com servers to capture a screenshot of your site the same way comments shows you preview of URLs.', 'mainwp-jetpack-protect-extension' ); ?>
			</div>
			<div class="actions">
				<div class="ui ok button"><?php esc_html_e( 'Yes', 'mainwp-jetpack-protect-extension' ); ?></div>
				<div class="ui cancel button"><?php esc_html_e( 'No', 'mainwp-jetpack-protect-extension' ); ?></div>
			</div>
		</div>

		<script type="text/javascript">
			jQuery( document ).ready( function () {
				jQuery('#reset-jetpack-protect-sites-settings').on( 'click', function () {
					mainwp_confirm(__( 'Are you sure.' ), function(){
						jQuery('.mainwp_hide_wpmenu_checkboxes input[id^="mainwp_show_column_"]').prop( 'checked', false );
						//default columns
						var cols = ['site','url','actions','wp-core','plugins','themes','last_check_date','connected'];
						jQuery.each( cols, function ( index, value ) {
							jQuery('.mainwp_hide_wpmenu_checkboxes input[id="mainwp_show_column_' + value + '"]').prop( 'checked', true );
						} );
						jQuery('input[name=reset_jetpack_protect_sites_columns_order]').attr('value',1);
						jQuery('#submit-jetpack-protect-sites-settings').click();
					}, false, false, true );
					return false;
				});
			} );
		</script>
		<?php
	}
}
