<?php
/**
 * MainWP Jetpack Protect Extension
 *
 * This file handles Jetpack Protect interactions.
 *
 * @package MainWP/Extensions
 */

namespace MainWP\Extensions\JetpackProtect;

/**
 * Class MainWP_Jetpack_Protect_Extension
 */
class MainWP_Jetpack_Protect_Extension {

	/**
	 * @var string $version version info.
	 */
	public $version = '1.0';


	/**
	 * Protected static variable to hold the single instance of the class.
	 *
	 * @var mixed Default null
	 */
	public static $instance = null;

	/**
	 * Method get_instance()
	 *
	 * Create a new public static instance of
	 * MainWP_Jetpack_Protect_Extension().
	 *
	 * @return void $instance New public static Instance.
	 */
	public static function get_instance() {
		if ( null == self::$instance ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	/**
	 * MainWP_Jetpack_Protect_Extension constructor.
	 *
	 * Run each time the class is called.
	 * Initialize default tokens upon creation of the object.
	 */
	public function __construct() {

		add_action( 'init', array( &$this, 'localization' ) );
		add_action( 'admin_init', array( &$this, 'admin_init' ) );
		add_action( 'mainwp_delete_site', array( &$this, 'hook_delete_site' ), 10, 1 );
		add_filter( 'mainwp_sitestable_getcolumns', array( $this, 'manage_sites_column' ), 10 );
		add_filter( 'mainwp_sitestable_item', array( $this, 'manage_sites_item' ), 10 );
		add_filter( 'mainwp_monitoring_sitestable_getcolumns', array( $this, 'manage_sites_column' ), 10 );
		add_filter( 'mainwp_monitoring_sitestable_item', array( $this, 'manage_sites_item' ), 10 );
		add_filter( 'mainwp_sitestable_sortable_columns', array( &$this, 'hook_mainwp_sitestable_sortable_columns' ), 10, 2 );

		MainWP_Jetpack_Protect_DB::get_instance()->install();
		MainWP_Jetpack_Protect::get_instance();
		MainWP_Jetpack_Protect_Plugin::get_instance();
	}

	/**
	 * Initiates the admin page when loaded & sets initial Class Variables.
	 */
	public function admin_init( $hook ) {
		if ( isset( $_GET['page'] ) && ( 'Extensions-Mainwp-Jetpack-Protect-Extension' === $_GET['page'] || 'mainwp_tab' === $_GET['page'] || 'ManageSitesJetpackProtect' === $_GET['page'] || ( 'managesites' === $_GET['page'] && isset( $_GET['dashboard'] ) ) ) ) {
			wp_enqueue_style( 'mainwp-jetpack-protect-extension', MAINWP_JETPACK_PROTECT_PLUGIN_URL . 'css/mainwp-jetpack-protect-extension.css', array(), $this->version );
			wp_enqueue_script( 'mainwp-jetpack-protect-extension', MAINWP_JETPACK_PROTECT_PLUGIN_URL . 'js/mainwp-jetpack-protect-extension.js', array( 'mainwp' ), $this->version, true );
			wp_enqueue_script( 'mainwp-jetpack-protect-dashboard', MAINWP_JETPACK_PROTECT_PLUGIN_URL . 'js/mainwp-jetpack-protect-dashboard.js', array( 'mainwp-jetpack-protect-extension' ), $this->version, true );
			wp_localize_script( 'mainwp-jetpack-protect-extension', 'mainwp_jetpack_protect_loc', array( 'nonce' => wp_create_nonce( '_wpnonce_mainwp_jetpack_protect' ) ) );
		}
	}

	/**
	 * Method localization().
	 */
	public function localization() {
		load_plugin_textdomain( 'mainwp-jetpack-protect-extension', false, dirname( plugin_basename( __FILE__ ) ) . '/languages/' );
	}

	/**
	 * Method hook_sync_others_data().
	 *
	 * @param array $data Input data parameters.
	 * @param mixed $website Input website data.
	 *
	 * @return array $data parameters.
	 */
	public function hook_sync_others_data( $data ) {
		if ( ! is_array( $data ) ) {
			$data = array();
		}
		$data['sync_JetpackProtect'] = 1;
		return $data;
	}

	/**
	 * Method hook_site_synced().
	 *
	 * @param array $website website object.
	 * @param array $information array data.
	 *
	 * @return void
	 */
	public function hook_site_synced( $website, $information = array() ) {
		$website_id = $website->id;
		if ( is_array( $information ) && isset( $information['sync_JetpackProtect_Data'] ) && is_array( $information['sync_JetpackProtect_Data'] ) ) {
			$data        = $information['sync_JetpackProtect_Data'];
			$data_status = isset( $data['status'] ) ? $data['status'] : array();
			MainWP_Jetpack_Protect_Utility::log_debug(is_array($data_status) ? error_log(print_r($data_status, true  )) : ( is_string($data_status) ? $data_status : '') );
			$updated     = false;
			if ( ! empty( $data_status ) && is_array( $data_status ) ) {
				$last_checked  = isset( $data_status['last_checked'] ) ? $data_status['last_checked'] : '';
				$lasttime_scan = 0;
				if ( ! empty( $last_checked ) ) {
					$lasttime_scan = strtotime( $last_checked );
				}
				if ( ! empty( $lasttime_scan ) ) {

					$current = MainWP_Jetpack_Protect_DB::get_instance()->get_protect_by( 'wpid', $website_id );
					if ( empty( $current ) || intval( $current->lasttime_scan ) <= intval( $lasttime_scan ) ) {

						$updated = true;

						$num_vuln          = isset( $data_status['num_threats'] ) ? intval( $data_status['num_threats'] ) : 0;
						$num_plugins_vuln  = isset( $data_status['num_plugins_threats'] ) ? intval( $data_status['num_plugins_threats'] ) : 0;
						$num_themes_vuln   = isset( $data_status['num_themes_threats'] ) ? intval( $data_status['num_themes_threats'] ) : 0;
						$num_files_vuln    = isset( $data_status['files'] ) && isset( $data_status['files'] ) && is_array( $data_status['files'] ) ? count( $data_status['files'] ) : 0;
						$num_database_vuln = isset( $data_status['database'] ) && isset( $data_status['database'] ) && is_array( $data_status['database'] ) ? count( $data_status['database'] ) : 0;
						$num_core_vuln     = isset( $data_status['core'] ) && isset( $data_status['core']['threats'] ) && is_array( $data_status['core']['threats'] ) ? count( $data_status['core']['threats'] ) : 0;

						$update = array(
							'wpid'                         => $website_id,
							'connected'                    => ! empty( $data['connected'] ) ? 1 : 0,
							'is_scan_active'                    => isset( $data['is_active'] ) && ! empty( $data['is_active'] ) ? 1 : 0,
							'lasttime_scan'                => $lasttime_scan,
							'num_vulnerabilities'          => $num_vuln,
							'num_core_vulnerabilities'     => $num_core_vuln,
							'num_plugins_vulnerabilities'  => $num_plugins_vuln,
							'num_themes_vulnerabilities'   => $num_themes_vuln,
							'num_files_vulnerabilities'    => $num_files_vuln,
							'num_database_vulnerabilities' => $num_database_vuln,
						);
						MainWP_Jetpack_Protect_DB::get_instance()->update_protect( $update );
						$last_status = MainWP_Jetpack_Protect_DB::get_instance()->get_protect_status_by( 'wpid', $website_id, 1 );
						if ( empty( $last_status ) || intval( $last_status->scan_time ) < intval( $lasttime_scan ) ) {
							$update = array(
								'wpid'        => $website_id,
								'status_data' => wp_json_encode( $data_status ),
								'scan_time'   => $lasttime_scan,
							);
							MainWP_Jetpack_Protect_DB::get_instance()->update_protect_status( $update );
						} elseif ( $lasttime_scan && $lasttime_scan == $last_status->scan_time ) {
							$update = array(
								'status_data' => wp_json_encode( $data_status ),
								'statusid'    => $last_status->statusid,
							);
							MainWP_Jetpack_Protect_DB::get_instance()->update_protect_status( $update );
						}
					}
				}
			}

			if ( ! $updated && isset( $data['connected'] ) ) {
				$update = array(
					'wpid'      => $website_id,
					'connected' => empty( $data['connected'] ) ? 0 : 1,
				);
				MainWP_Jetpack_Protect_DB::get_instance()->update_protect( $update );
			}
		}
	}

	/**
	 * Method hook_delete_site().
	 */
	public function hook_delete_site( $website ) {
		if ( $website ) {
			MainWP_Jetpack_Protect_DB::get_instance()->delete_protect_by( 'wpid', $website->id );
			MainWP_Jetpack_Protect_DB::get_instance()->delete_protect_status_by( 'wpid', $website->id );
		}
	}

	/**
	 * Manage Sites Column
	 *
	 * Adds the custom column in the Manage Sites and Monitoring tables.
	 *
	 * @param array $columns Table comlumns.
	 *
	 * @return array $columns Table comlumns.
	 */
	public function manage_sites_column( $columns ) {
		$columns['jetpack_protect'] = __( 'Jetpack Protect', 'mainwp-lighthouse-extension' );
		return $columns;
	}

	/**
	 * Manage Sites Item
	 *
	 * Adds the custom column data in the Manage Sites and Monitoring tables.
	 *
	 * @param array $item Site comlumn data.
	 *
	 * @return array $item Site comlumn data.
	 */
	public function manage_sites_item( $item ) {
		$jp_protect = MainWP_Jetpack_Protect_DB::get_instance()->get_protect_by( 'wpid', $item['id'] );
		if ( isset( $jp_protect->lasttime_scan ) && '' != $jp_protect->lasttime_scan ) {
			$data_tooltip            = 'Latest result as of: ' . MainWP_Jetpack_Protect_Utility::format_timestamp( MainWP_Jetpack_Protect_Utility::get_timestamp( $jp_protect->lasttime_scan ) ) . '   |   WP: ' . $jp_protect->num_core_vulnerabilities . '   |   Plugins: ' . $jp_protect->num_plugins_vulnerabilities . '   |   Themes: ' . $jp_protect->num_themes_vulnerabilities;
			$item['jetpack_protect'] = '<a href="admin.php?page=ManageSitesJetpackProtect&id=' . $item['id'] . '" data-tooltip="' . $data_tooltip . '" data-position="left center" data-inverted="" class="ui basic ' . ( $jp_protect->num_vulnerabilities > 0 ? 'red' : '' ) . ' label">' . $jp_protect->num_vulnerabilities . '</a>';
		}

		return $item;
	}

	/**
	 * Hook hook_mainwp_sitestable_sortable_columns.
	 *
	 * Hook sites table sortable columns.
	 *
	 * @return mixed $sortable_columns.
	 */
	public function hook_mainwp_sitestable_sortable_columns( $sortable_columns ) {
		$sortable_columns['jetpack_protect'] = array( 'jetpack_protect', false );
		return $sortable_columns;
	}

}
