<?php
/**
 * MainWP Jetpack Protect DB
 *
 * This file handles DB interactions.
 *
 * @package MainWP/Extensions
 */

namespace MainWP\Extensions\JetpackProtect;

/**
 * Class MainWP_Jetpack_Protect_DB
 */
class MainWP_Jetpack_Protect_DB {

	/**
	 * Private variable to hold the database version info.
	 *
	 * @var string DB version info.
	 */
	private $db_version = '1.1.0';


	/**
	 * Table prefix.
	 *
	 * @var string $table_prefix
	 */
	protected $table_prefix;

	/**
	 * WordPress Database.
	 *
	 * @var mixed $wpdb WordPress Database.
	 */
	protected $wpdb;

	/**
	 * Protected static variable to hold the single instance of the class.
	 *
	 * @var mixed Default null
	 */
	private static $instance = null;

	/**
	 * MainWP_Jetpack_Protect_DB class constructor.
	 *
	 * @return void
	 */
	public function __construct() {
		global $wpdb;
		$this->table_prefix = $wpdb->prefix . 'mainwp_';
		$this->wpdb         = &$wpdb;
	}

	/**
	 * Method get_instance()
	 *
	 * Create a new public static instance of
	 * MainWP_Jetpack_Protect_DB().
	 *
	 * @return void $instance New public static Instance.
	 */
	public static function get_instance() {
		if ( null == self::$instance ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	public function table_name( $suffix ) {
		return $this->table_prefix . $suffix;
	}

	public static function use_mysqli() {
		if ( ! function_exists( '\mysqli_connect' ) ) {
			return false;
		}
		return ( self::$instance->wpdb->dbh instanceof \mysqli );
	}

	// Installs new DB
	public function install() {

		$currentVersion = get_option( 'mainwp_jetpack_protect_db_version', '' );

		if ( version_compare( $currentVersion, $this->db_version, '>=' ) ) {
			return;
		}

		$charset_collate = $this->wpdb->get_charset_collate();

		$sql = array();

		$tbl = 'CREATE TABLE `' . $this->table_name( 'jetpack_protect' ) . '` (
`id` int(11) NOT NULL AUTO_INCREMENT,
`wpid` int(11) NOT NULL,
`lasttime_scan` int(11) NOT NULL DEFAULT 0,
`num_vulnerabilities` int(11) NOT NULL DEFAULT 0,
`num_core_vulnerabilities` int(11) NOT NULL DEFAULT 0,
`num_plugins_vulnerabilities` int(11) NOT NULL DEFAULT 0,
`num_themes_vulnerabilities` int(11) NOT NULL DEFAULT 0,
`num_files_vulnerabilities` int(11) NOT NULL DEFAULT 0,
`num_database_vulnerabilities` int(11) NOT NULL DEFAULT 0,
`hide_plugin` tinyint(1) NOT NULL DEFAULT 0,
`connected` tinyint(1) NOT NULL DEFAULT 0,
`is_scan_active` tinyint(1) NOT NULL DEFAULT 0,
UNIQUE KEY wpid (wpid)';
		if ( '' == $currentVersion ) {
			$tbl .= ',
PRIMARY KEY  (`id`)  ';
		}

		$tbl  .= ') ' . $charset_collate;
		$sql[] = $tbl;

		$tbl = 'CREATE TABLE `' . $this->table_name( 'jetpack_protect_status' ) . '` (
`statusid` int(11) NOT NULL AUTO_INCREMENT,
`wpid` int(11) NOT NULL,
`status_data` longtext NOT NULL DEFAULT "",
`scan_time` int(11) NOT NULL DEFAULT 0';

		if ( '' == $currentVersion || version_compare( $currentVersion, '1.0.3', '<' ) ) {
			$tbl .= ',
PRIMARY KEY  (`statusid`)  ';
		}

		$tbl  .= ') ' . $charset_collate;
		$sql[] = $tbl;

		// error_reporting( 0 ); // to disable any error output
		require_once ABSPATH . 'wp-admin/includes/upgrade.php';
		foreach ( $sql as $query ) {
			dbDelta( $query );
		}

		update_option( 'mainwp_jetpack_protect_db_version', $this->db_version );

	}

	public function update_protect( $data ) {

		$id   = 0;
		$wpid = 0;

		if ( isset( $data['id'] ) ) {
			$id = intval( $data['id'] );
			unset( $data['id'] );
		}

		if ( isset( $data['wpid'] ) ) {
			$wpid = intval( $data['wpid'] );
		}

		if ( ! empty( $id ) ) {
			if ( isset( $data['wpid'] ) ) {
				unset( $data['wpid'] );
			}
			if ( ! empty( $data ) ) {
				$this->wpdb->update( $this->table_name( 'jetpack_protect' ), $data, array( 'id' => intval( $id ) ) );
			}
			return $this->get_protect_by( 'id', $id );
		} elseif ( ! empty( $wpid ) ) {
			$current = $this->get_protect_by( 'wpid', $wpid );
			if ( ! empty( $current ) ) {
				if ( isset( $data['wpid'] ) ) {
					unset( $data['wpid'] ); // not update this field.
				}
				$this->wpdb->update( $this->table_name( 'jetpack_protect' ), $data, array( 'wpid' => intval( $wpid ) ) );
			} else {
				$this->wpdb->insert( $this->table_name( 'jetpack_protect' ), $data );
			}
			return $this->get_protect_by( 'wpid', $wpid );
		} else {
			if ( $this->wpdb->insert( $this->table_name( 'jetpack_protect' ), $data ) ) {
				return $this->get_protect_by( 'id', $this->wpdb->insert_id );
			}
		}
		return false;
	}

	public function update_protect_status( $data ) {
		$statusid = 0;

		if ( isset( $data['statusid'] ) && $data['statusid'] ) {
			$statusid = intval( $data['statusid'] );
			unset( $data['statusid'] );
		}

		if ( ! empty( $statusid ) ) {
			if ( ! empty( $data ) ) {
				$this->wpdb->update( $this->table_name( 'jetpack_protect_status' ), $data, array( 'statusid' => intval( $statusid ) ) );
			}
			return $this->get_protect_status_by( 'statusid', $statusid );
		} else {
			if ( $this->wpdb->insert( $this->table_name( 'jetpack_protect_status' ), $data ) ) {
				return $this->get_protect_status_by( 'statusid', $this->wpdb->insert_id );
			}
		}
		return false;
	}

	public function get_protect_by( $by = 'id', $value = null ) {
		if ( empty( $value ) ) {
			return false;
		}

		$sql = '';
		if ( 'id' == $by ) {
			$sql = $this->wpdb->prepare( 'SELECT * FROM ' . $this->table_name( 'jetpack_protect' ) . ' WHERE `id` =%d', $value );
		} elseif ( 'wpid' == $by ) {
			$sql = $this->wpdb->prepare( 'SELECT * FROM ' . $this->table_name( 'jetpack_protect' ) . ' WHERE `wpid` = %d', $value );
		}

		if ( ! empty( $sql ) ) {
			return $this->wpdb->get_row( $sql );
		}

		return false;
	}

	public function get_protect_status_by( $by = 'statusid', $value = null, $limit = false, $last_scan_time = false ) {
		if ( empty( $value ) ) {
			return false;
		}

		$sql_limit = '';
		if ( ! empty( $limit ) ) {
			$sql_limit = ' LIMIT ' . intval( $limit );
		}

		$order_by = ' ORDER BY scan_time DESC ';

		$sql = '';
		if ( 'statusid' == $by ) {
			$sql = $this->wpdb->prepare( 'SELECT * FROM ' . $this->table_name( 'jetpack_protect_status' ) . ' WHERE `statusid` =%d', $value );
		} elseif ( 'wpid' == $by ) {
			if ( ! empty( $last_scan_time ) ) {
				$sql_limit = ' LIMIT 1 ';
				$sql       = $this->wpdb->prepare( 'SELECT * FROM ' . $this->table_name( 'jetpack_protect_status' ) . ' WHERE `wpid` = %d AND scan_time <= %d' . $order_by . $sql_limit, $value, $last_scan_time );
			} else {
				$sql = $this->wpdb->prepare( 'SELECT * FROM ' . $this->table_name( 'jetpack_protect_status' ) . ' WHERE `wpid` = %d ' . $order_by . $sql_limit, $value );
			}
		}

		if ( ! empty( $sql ) ) {
			return $this->wpdb->get_row( $sql );
		}

		return false;
	}


	public function get_protect_status_last_sites() {
		$sql = '
SELECT ps.statusid,ps.wpid,ps.status_data,ps.scan_time 
FROM ' . $this->table_name( 'jetpack_protect_status' ) . ' ps 
JOIN ( SELECT DISTINCT psd.wpid, ( SELECT ps2.statusid FROM ' . $this->table_name( 'jetpack_protect_status' ) . ' ps2 WHERE psd.wpid = ps2.wpid ORDER BY ps2.scan_time DESC LIMIT 1 ) as statusid FROM wp_mainwp_jetpack_protect_status psd ) psint 
ON ps.statusid = psint.statusid
WHERE 1 ORDER BY ps.scan_time DESC';
		return $this->wpdb->get_results( $sql );
	}

	/**
	 * Get sql to query websites and extension data.
	 *
	 * @param int|false $site_id Website ID.
	 *
	 * @return string sql to query.
	 */
	public function get_sql_websites_ext_data( $site_id = false ) {

		global $mainJetpackProtectExtensionActivator;
		
		$where = '';
		if ( $site_id ) {
			$where = ' jp.wpid = ' . intval( $site_id );
		}

		$join = ' LEFT JOIN ' . $this->table_name( 'jetpack_protect' ) . ' jp ON wp.id = jp.wpid ';

		$params    = array(
			'extra_select_wp_fields'  => array( 'plugin_upgrades', 'plugins', 'themes', 'theme_upgrades' ),
			'extra_select_sql_fields' => ' jp.id as protect_id,  jp.connected, jp.is_scan_active, jp.hide_plugin, jp.lasttime_scan, jp.num_vulnerabilities, jp.num_core_vulnerabilities, jp.num_plugins_vulnerabilities, jp.num_themes_vulnerabilities, jp.num_files_vulnerabilities, jp.num_database_vulnerabilities  ',
			'extra_join'              => $join,
			'extra_where'             => $where,
		);
		$sql_sites = apply_filters( 'mainwp_getsqlwebsites_for_current_user', false, $mainJetpackProtectExtensionActivator->get_child_file(), $mainJetpackProtectExtensionActivator->get_child_key(), $params );
		return $sql_sites;
	}

	public function delete_protect_by( $by = 'id', $value = null ) {
		if ( 'id' === $by ) {
			if ( $this->wpdb->query( $this->wpdb->prepare( 'DELETE FROM ' . $this->table_name( 'jetpack_protect' ) . ' WHERE id=%d ', $value ) ) ) {
				return true;
			}
		} elseif ( 'wpid' === $by ) {
			if ( $this->wpdb->query( $this->wpdb->prepare( 'DELETE FROM ' . $this->table_name( 'jetpack_protect' ) . ' WHERE wpid=%d ', $value ) ) ) {
				return true;
			}
		}
		return false;
	}

	public function delete_protect_status_by( $by = 'id', $value = null ) {
		if ( 'id' === $by ) {
			if ( $this->wpdb->query( $this->wpdb->prepare( 'DELETE FROM ' . $this->table_name( 'jetpack_protect_status' ) . ' WHERE id=%d ', $value ) ) ) {
				return true;
			}
		} elseif ( 'wpid' === $by ) {
			if ( $this->wpdb->query( $this->wpdb->prepare( 'DELETE FROM ' . $this->table_name( 'jetpack_protect_status' ) . ' WHERE wpid=%d ', $value ) ) ) {
				return true;
			}
		}
		return false;
	}


	/**
	 * Escape data.
	 *
	 * @param array $data Data to escape.
	 *
	 * @return array Returns escaped data.
	 */
	protected function escape( $data ) {
		if ( function_exists( 'esc_sql' ) ) {
			return esc_sql( $data );
		} else {
			return $this->wpdb->escape( $data ); }
	}

	/**
	 * SQL Query.
	 *
	 * @param string $sql SQL query string.
	 *
	 * @return mixed $result Return FALSE on failure & query results on success.
	 */
	public function query( $sql ) {
		if ( null == $sql ) {
			return false;
		}

		$result = self::m_query( $sql, $this->wpdb->dbh );

		if ( ! $result || ( 0 == self::num_rows( $result ) ) ) {
			return false; }
		return $result;
	}

	/**
	 * MySQLi Query.
	 *
	 * @param string $query Query string.
	 * @param string $link  Query link.
	 *
	 * @return mixed FALSE on failure & query results on success.
	 */
	public static function m_query( $query, $link ) {
		if ( self::use_mysqli() ) {
			return \mysqli_query( $link, $query );
		} else {
			return \mysql_query( $query, $link );
		}
	}

	/**
	 * Returns the current row of a result set, then print each field's value.
	 *
	 * @param string $result Required. Specifies a result set identifier.
	 *
	 * @return mixed Return fetched object or Null.
	 */
	public static function fetch_object( $result ) {
		if ( self::use_mysqli() ) {
			return \mysqli_fetch_object( $result );
		} else {
			return \mysql_fetch_object( $result );
		}
	}

	/**
	 * Fetch rows from a result-set, then free the memory associated with the result.
	 *
	 * @param string $result Required. Specifies a result set identifier.
	 *
	 * @return object Return fetched object or Null.
	 */
	public static function free_result( $result ) {
		if ( self::use_mysqli() ) {
			return \mysqli_free_result( $result );
		} else {
			return \mysql_free_result( $result );
		}
	}

	/**
	 * Seek to row offset in the result-set.
	 *
	 * @param string $result Required. Specifies a result set identifier.
	 * @param int    $offset Required. Specifies the field offset. Must be between 0 and the total number of rows - 1.
	 *
	 * @return bool TRUE on success. FALSE on failure.
	 */
	public static function data_seek( $result, $offset ) {
		if ( self::use_mysqli() ) {
			return \mysqli_data_seek( $result, $offset );
		} else {
			return \mysql_data_seek( $result, $offset );
		}
	}

	/**
	 * Fetch a result row as a numeric array and as an associative array.
	 *
	 * @param string $result      Required. Specifies a result set identifier.
	 * @param null   $result_type Optional. Specifies what type of array that should be produced.
	 *
	 * @return mixed Returns an array of strings that corresponds to the fetched row. NULL if there are no more rows in result-set.
	 */
	public static function fetch_array( $result, $result_type = null ) {
		if ( self::use_mysqli() ) {
			return \mysqli_fetch_array( $result, ( null == $result_type ? MYSQLI_BOTH : $result_type ) );
		} else {
			return \mysql_fetch_array( $result, ( null == $result_type ? MYSQL_BOTH : $result_type ) );
		}
	}

	/**
	 * Returns the number of rows in a result set.
	 *
	 * @param string $result Required. Specifies a result set identifier.
	 *
	 * @return int Returns the number of rows in the result set or FALSE on failure.
	 */
	public static function num_rows( $result ) {
		if ( self::use_mysqli() ) {
			return \mysqli_num_rows( $result );
		} else {
			return \mysql_num_rows( $result );
		}
	}

	/**
	 * Checks if there are results.
	 *
	 * @param string $result Required. Specifies a result set identifier.
	 *
	 * @return mixed Return FALSE on failure, Instance of mysqli_result or resource.
	 */
	public static function is_result( $result ) {
		if ( self::use_mysqli() ) {
			return ( $result instanceof \mysqli_result );
		} else {
			return is_resource( $result );
		}
	}

	/**
	 * Executes a SQL query and returns the entire SQL result.
	 *
	 * @param string $sql SQL query string.
	 *
	 * @return mixed Database query results.
	 */
	public function get_results_result( $sql ) {
		if ( null == $sql ) {
			return null;
		}
		return $this->wpdb->get_results( $sql, OBJECT_K );
	}
}
