<?php
/*
 * Plugin Name: MainWP Database Updater Extension
 * Plugin URI: https://mainwp.com
 * Description: MainWP Database Updater Extension is designed for updating the databases of two popular WordPress plugins - WooCommerce and Elementor. With this extension, users can easily and quickly update the databases of these plugins without having to do so manually.
 * Version: 5.0.7
 * Author: MainWP
 * Author URI: https://mainwp.com
 * Documentation URI: https://mainwp.com/kb/mainwp-database-updater-extension/
 * Icon URI:
 */

namespace MainWP\Extensions\DatabaseUpdater;

if ( ! defined( 'MAINWP_DATABASE_UPDATER_PLUGIN_URL' ) ) {
	define( 'MAINWP_DATABASE_UPDATER_PLUGIN_URL', plugin_dir_url( __FILE__ ) );
}

if ( ! defined( 'MAINWP_DATABASE_UPDATER_LOG_PRIORITY' ) ) {
	define( 'MAINWP_DATABASE_UPDATER_LOG_PRIORITY', 65 );
}

if ( ! defined( 'MAINWP_DATABASE_UPDATER_PLUGIN_SLUG' ) ) {
	define( 'MAINWP_DATABASE_UPDATER_PLUGIN_SLUG', plugin_basename( __FILE__ ) );
}

class MainWP_Database_Updater_Extension_Activator {

	protected $mainwpMainActivated = false;
	protected $childEnabled        = false;
	protected $childKey            = false;
	protected $childFile;
	protected $plugin_handle    = 'mainwp-database-updater-extension';
	protected $product_id       = 'MainWP Database Updater Extension';
	protected $software_version = '5.0.7';

	public function __construct() {
		$this->childFile = __FILE__;

		spl_autoload_register( array( $this, 'autoload' ) );
		register_activation_hook( __FILE__, array( $this, 'activate' ) );
		register_deactivation_hook( __FILE__, array( $this, 'deactivate' ) );

		add_filter( 'mainwp_getextensions', array( &$this, 'get_this_extension' ) );
		$this->mainwpMainActivated = apply_filters( 'mainwp_activated_check', false );

		if ( $this->mainwpMainActivated !== false ) {
			$this->activate_this_plugin();
		} else {
			add_action( 'mainwp_activated', array( &$this, 'activate_this_plugin' ) );
		}
		add_action( 'admin_notices', array( &$this, 'mainwp_error_notice' ) );
	}


	/**
	 * Loads class automatically.
	 *
	 * @param string $class_name Class name.
	 *
	 * @return void
	 */
	public function autoload( $class_name ) {

		if ( 0 === strpos( $class_name, 'MainWP\Extensions\DatabaseUpdater' ) ) {
			// trip the namespace prefix: MainWP\Extensions\DatabaseUpdater\.
			$class_name = str_replace( 'MainWP\Extensions\DatabaseUpdater\\', '', $class_name );
		} else {
			return;
		}

		if ( 0 !== strpos( $class_name, 'MainWP_Database_Updater_' ) ) {
			return;
		}

		$class_name = strtolower( str_replace( '_', '-', $class_name ) );

		$class_file = WP_PLUGIN_DIR . DIRECTORY_SEPARATOR . str_replace( basename( __FILE__ ), '', plugin_basename( __FILE__ ) ) . 'class' . DIRECTORY_SEPARATOR . 'class-' . $class_name . '.php';
		if ( file_exists( $class_file ) ) {
			require_once $class_file;
		}
	}

	/**
	 * Add your extension to MainWP via the 'mainwp_getextensions' filter.
	 *
	 * @param array $pArray Array containing the extensions info.
	 *
	 * @return array $pArray Updated array containing the extensions info.
	 */
	public function get_this_extension( $pArray ) {
		$pArray[] = array(
			'plugin'     => __FILE__,
			'api'        => $this->plugin_handle,
			'mainwp'     => true,
			'apiManager' => true,
			'href'       => 'admin.php?page=UpdatesManage&tab=plugin-db-updates',
		);
		return $pArray;
	}

	/**
	 * Displays the extension page with adequate header and footer.
	 */
	public function settings() {
		do_action( 'mainwp_pageheader_extensions', __FILE__ );
		do_action( 'mainwp_pagefooter_extensions', __FILE__ );
	}

	/**
	 * Activate the extension API license and initiate the extension.
	 */
	public function activate_this_plugin() {
		$this->mainwpMainActivated = apply_filters( 'mainwp_activated_check', $this->mainwpMainActivated );
		$this->childEnabled        = apply_filters( 'mainwp_extension_enabled_check', __FILE__ );
		$this->childKey            = $this->childEnabled['key'];
		if ( function_exists( '\mainwp_current_user_can' ) && ! \mainwp_current_user_can( 'extension', 'mainwp-database-updater-extension' ) ) {
			return;
		}
		MainWP_Database_Updater_Admin::get_instance();
		MainWP_Database_Updater_Overview::get_instance();

	}

	/**
	 * Get the extenion key.
	 *
	 * @return string
	 */
	public function get_child_key() {
		return $this->childKey;
	}

	/**
	 * Get the extension file.
	 *
	 * @return string
	 */
	public function get_child_file() {
		return $this->childFile;
	}

	/**
	 * Render the warning notice if the MainWP Dashboard plugin is not activated.
	 */
	public function mainwp_error_notice() {
		global $current_screen;
		if ( $current_screen->parent_base == 'plugins' && $this->mainwpMainActivated == false ) {
			echo '<div class="error"><p>MainWP Database Updater Extension ' . sprintf( esc_html__( 'requires %1$sMainWP Dashboard Plugin%2$s to be activated in order to work. Please install and activate %3$sMainWP Dashboard Plugin%4$s first.', 'mainwp-database-updater-extension' ), '<a href="https://mainwp.com/" target="_blank">', '</a>', '<a href="https://mainwp.com/" target="_blank">', '</a>' ) . '</p></div>';
		}
	}

	/**
	 * Activate the extension license.
	 */
	public function activate() {
		$options = array(
			'product_id'       => $this->product_id,
			'software_version' => $this->software_version,
		);
		do_action( 'mainwp_activate_extention', $this->plugin_handle, $options );
	}

	/**
	 * Deactivate the extension license.
	 */
	public function deactivate() {
		do_action( 'mainwp_deactivate_extention', $this->plugin_handle );
	}
}

global $mainWPDatabaseUpdaterExtensionActivator;
$mainWPDatabaseUpdaterExtensionActivator = new MainWP_Database_Updater_Extension_Activator();
