<?php

/**
 * Class MainWP_Database_Updater_Utility
 */
namespace MainWP\Extensions\DatabaseUpdater;

class MainWP_Database_Updater_Utility {

	public static $instance = null;

	/**
	 * Get Instance
	 *
	 * Creates public static instance.
	 *
	 * @static
	 *
	 * @return MainWP_Database_Updater_Utility
	 */
	public static function get_instance() {
		if ( null == self::$instance ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	public function __construct() {
	}

	/**
	 * Method get_sql_websites_for_current_user().
	 *
	 * Gets sql websites for current user.
	 *
	 * @param array $params query params.
	 *
	 * @return string sql query.
	 */
	public static function get_sql_websites_for_current_user( $params = array() ) {
		global $mainWPDatabaseUpdaterExtensionActivator;
		return apply_filters( 'mainwp_getsqlwebsites_for_current_user', false, $mainWPDatabaseUpdaterExtensionActivator->get_child_file(), $mainWPDatabaseUpdaterExtensionActivator->get_child_key(), $params );
	}

	/**
	 * Method query().
	 *
	 * Handle db query.
	 *
	 * @param string $sql query.
	 *
	 * @return mixed results.
	 */
	public static function db_query( $sql ) {
		return apply_filters( 'mainwp_db_query', false, $sql );
	}

	/**
	 * Method fetch_object().
	 *
	 * Handle fetch object db.
	 *
	 * @param mixed $websites websites results.
	 *
	 * @return mixed results.
	 */
	public static function fetch_object( $websites ) {
		return apply_filters( 'mainwp_db_fetch_object', false, $websites );
	}

	/**
	 * Method data_seek().
	 *
	 * Handle data seek.
	 *
	 * @param mixed $websites websites results.
	 * @param int   $offset Offset.
	 *
	 * @return mixed results.
	 */
	public static function data_seek( $websites, $offset = 0 ) {
		do_action( 'mainwp_db_data_seek', $websites, $offset );
	}

	/**
	 * Method free_result().
	 *
	 * Handle fetch result db.
	 *
	 * @param mixed $results websites results.
	 *
	 * @return mixed results.
	 */
	public static function free_result( $results ) {
		if ( empty( $results ) ) {
			return;
		}
		return apply_filters( 'mainwp_db_free_result', false, $results );
	}

	/**
	 * Get timestamp.
	 *
	 * @param string $timestamp Holds Timestamp.
	 *
	 * @return float|int Return GMT offset.
	 */
	public static function get_timestamp( $timestamp = false ) {
		if ( empty( $timestamp ) ) {
			$timestamp = time();
		}
		$gmtOffset = get_option( 'gmt_offset' );

		return ( $gmtOffset ? ( $gmtOffset * HOUR_IN_SECONDS ) + $timestamp : $timestamp );
	}

	/**
	 * Format timestamp.
	 *
	 * @param string $timestamp Holds Timestamp.
	 * @param bool   $gmt Whether to set as General mountain time. Default: FALSE.
	 *
	 * @return string Return Timestamp.
	 */
	public static function format_timestamp( $timestamp, $gmt = false ) {
		return date_i18n( get_option( 'date_format' ) . ' ' . get_option( 'time_format' ), $timestamp, $gmt );
	}

	/**
	 * Format datestamp.
	 *
	 * @param string $timestamp Holds Timestamp.
	 * @param bool   $gmt Whether to set as General mountain time. Default: FALSE.
	 *
	 * @return string Return Timestamp.
	 */
	public static function format_datestamp( $timestamp, $gmt = false ) {
		return date_i18n( get_option( 'date_format' ), $timestamp, $gmt );
	}

	/**
	 * Format date.
	 *
	 * @param string $timestamp Holds Timestamp.
	 * @param bool   $gmt Whether to set as General mountain time. Default: FALSE.
	 *
	 * @return string Return Timestamp.
	 */
	public static function format_date( $timestamp, $gmt = false ) {
		return date_i18n( get_option( 'date_format' ), $timestamp, $gmt );
	}


	/**
	 * Method map_fields()
	 *
	 * Map Site.
	 *
	 * @param mixed $website Website to map.
	 * @param mixed $keys Keys to map.
	 * @param bool  $object_output Output format array|object.
	 *
	 * @return object $outputSite Mapped site.
	 */
	public static function map_fields( &$website, $keys, $object_output = false ) {
		$outputSite = array();
		if ( ! empty( $website ) ) {
			if ( is_object( $website ) ) {
				foreach ( $keys as $key ) {
					if ( property_exists( $website, $key ) ) {
						$outputSite[ $key ] = $website->$key;
					}
				}
			} elseif ( is_array( $website ) ) {
				foreach ( $keys as $key ) {
					$outputSite[ $key ] = $website[ $key ];
				}
			}
		}

		if ( $object_output ) {
			return (object) $outputSite;
		} else {
			return $outputSite;
		}
	}


	/**
	 * Get Websites
	 *
	 * Gets all child sites through the 'mainwp_getsites' filter.
	 *
	 * @param array|null $site_id  Child sites ID.
	 *
	 * @return array Child sites array.
	 */
	public static function get_websites( $site_id = null ) {
		global $mainWPDatabaseUpdaterExtensionActivator;
		return apply_filters( 'mainwp_getsites', $mainWPDatabaseUpdaterExtensionActivator->get_child_file(), $mainWPDatabaseUpdaterExtensionActivator->get_child_key(), $site_id, false );
	}

	/**
	 * Get DB Websites
	 *
	 * Gets all child sites through the 'mainwp_getsites' filter.
	 *
	 * @param array $site_ids  Child sites IDs.
	 * @param array $group_ids Groups IDs.
	 *
	 * @return array Child sites array.
	 */
	public static function get_db_websites( $site_ids = null, $group_ids = false ) {
		global $mainWPDatabaseUpdaterExtensionActivator;
		return apply_filters( 'mainwp_getdbsites', $mainWPDatabaseUpdaterExtensionActivator->get_child_file(), $mainWPDatabaseUpdaterExtensionActivator->get_child_key(), $site_ids, $group_ids );
	}

	/**
	 * Method update_website_option().
	 *
	 * Update the website option.
	 *
	 * @param object|int $website website object or ID.
	 * @param string     $opt_name website.
	 * @param string     $opt_value website.
	 */
	public static function update_website_option( $website, $opt_name, $opt_value ) {
		return apply_filters( 'mainwp_updatewebsiteoptions', false, $website, $opt_name, $opt_value );
	}

	/**
	 * Method get_website_options().
	 *
	 * Get the website options.
	 *
	 * @param object|int   $website website object or ID.
	 * @param string|array $names Options name.
	 */
	public static function get_website_options( $website, $names ) {
		return apply_filters( 'mainwp_getwebsiteoptions', false, $website, $names );
	}


	/**
	 * Method array_sort()
	 *
	 * Sort given array by given flags.
	 *
	 * @param mixed  $array Array to sort.
	 * @param mixed  $key Array key.
	 * @param string $sort_flag Flags to sort by. Default = SORT_STRING.
	 */
	public static function array_sort( &$array, $key, $sort_flag = SORT_STRING ) {
		$sorter = array();
		$ret    = array();
		reset( $array );
		foreach ( $array as $ii => $val ) {
			$sorter[ $ii ] = $val[ $key ];
		}
		asort( $sorter, $sort_flag );
		foreach ( $sorter as $ii => $val ) {
			$ret[ $ii ] = $array[ $ii ];
		}
		$array = $ret;
	}

	/**
	 * Debugging log info.
	 *
	 * Sets logging for debugging purpose
	 *
	 * @param string $message Log info message.
	 */
	public static function log_info( $message ) {
		self::log_debug( $message, 2 );
	}

	/**
	 * Debugging log.
	 *
	 * Sets logging for debugging purpose.
	 *
	 * @param string $message Log debug message.
	 */
	public static function log_debug( $message, $log_color = 3 ) {
		$cron = '';
		if ( defined( 'DOING_CRON' ) && DOING_CRON ) {
			$cron = 'CRON :: ';
		}
		// Set color: 0 - LOG, 1 - WARNING, 2 - INFO, 3- DEBUG.

		if ( ! in_array( $log_color, array( 0, 1, 2, 3 ) ) ) {
			$log_color = 3;
		}
		do_action( 'mainwp_log_action', 'Database Updater :: ' . $cron . $message, MAINWP_DATABASE_UPDATER_LOG_PRIORITY, $log_color );
	}


	/**
	 * Show Info Messages
	 *
	 * Check whenther or not to show the MainWP Message.
	 *
	 * @param string $notice_id Notice ID.
	 *
	 * @return bool False if hidden, true to show.
	 */
	public static function show_mainwp_message( $notice_id ) {
		$status = get_user_option( 'mainwp_notice_saved_status' );
		if ( ! is_array( $status ) ) {
			$status = array();
		}
		if ( isset( $status[ $notice_id ] ) ) {
			return false;
		}
		return true;
	}


	/**
	 * Get Color Code
	 *
	 * Returns CSS class to use correct color for the element.
	 *
	 * @param int $value value.
	 *
	 * @return string CSS class.
	 */
	public static function color_code( $value ) {
		$color = '';
		if ( 5 < $value ) {
			$color = 'red';
		} elseif ( 0 < $value && 5 >= $value ) {
			$color = 'yellow';
		} else {
			$color = 'green';
		}
		return $color;
	}

}
