<?php

namespace MAINWP\BackUpWordPress;

require_once( MAINWP_HMBKP_PLUGIN_PATH . 'classes/class-setup.php' );

register_activation_hook( __FILE__, array( 'MAINWP\BackUpWordPress\Setup', 'activate' ) );
register_deactivation_hook( __FILE__, array( 'MAINWP\BackUpWordPress\Setup', 'deactivate' ) );

/**
 * Class Plugin
 */
final class Plugin {

	const PLUGIN_VERSION = '3.2.4';

	/**
	 * @var Plugin The singleton instance.
	 */
	private static $instance;

	/**
	 * Instantiates a new Plugin object.
	 */
	private function __construct() {
		add_action( 'plugins_loaded', array( $this, 'plugins_loaded' ) );
		//Add Javascripts script
	}

	/**
	 * Insures we always return the same object.
	 *
	 * @return Plugin
	 */
	public static function get_instance() {

		if ( ! ( self::$instance instanceof Plugin ) ) {
			self::$instance = new Plugin();
		}

		return self::$instance;
	}

	/**
	 * Initialize the plugin.
	 */
	public function plugins_loaded() {

		if ( true !== $this->maybe_self_deactivate() ) {

			$this->constants();

			$this->includes();

			$this->hooks();

			$this->text_domain();
		}
	}

	/**
	 * Check plugin requirements.
	 *
	 * @return bool True is fails requirements. False otherwise.
	 */
	public function maybe_self_deactivate() {

		if ( false === Setup::meets_requirements() ) {

			add_action( 'admin_init', array( 'MAINWP\BackUpWordPress\Setup', 'self_deactivate' ) );

			add_action( 'admin_notices', array( 'MAINWP\BackUpWordPress\Setup', 'display_admin_notices' ) );

			return true;
		}

		return false;
	}

	/**
	 * Define all the constants.
	 */
	public function constants() {

		if ( ! defined( 'MAINWP_HMBKP_PLUGIN_SLUG' ) ) {
			define( 'MAINWP_HMBKP_PLUGIN_SLUG', dirname( MAINWP_HMBKP_BASENAME ) );
		}

		if ( ! defined( 'MAINWP_HMBKP_PLUGIN_URL' ) ) {
			define( 'MAINWP_HMBKP_PLUGIN_URL', plugin_dir_url( MAINWP_HMBKP_BASENAME ) );
		}

		if ( ! defined( 'MAINWP_HMBKP_PLUGIN_LANG_DIR' ) ) {
			define( 'MAINWP_HMBKP_PLUGIN_LANG_DIR', apply_filters( 'mainwp_hmbkp_filter_lang_dir', MAINWP_HMBKP_PLUGIN_SLUG . '/languages/' ) );
		}

		if ( ! defined( 'MAINWP_HMBKP_EXTENSION_URL' ) ) {
			$siteId = 0;
			$page = admin_url( 'admin.php' );

			if ( \MainWP_BackUp_WordPress::is_managesites_page() ) {
				$page_slug = 'ManageSitesBackUpWordPress';
				$siteId = \MainWP_BackUp_WordPress::get_site_id_managesites_backupwp();
				$page = add_query_arg( array( 'id' => $siteId, 'page' => $page_slug ), $page );
			} else {
				$page_slug = 'Extensions-Mainwp-Backupwordpress-Extension';
				$page = add_query_arg( array( 'page' => $page_slug ), $page );
			}

			if ( $siteId ) {
				$page = add_query_arg( 'id', $siteId, $page );
			}
			define( 'MAINWP_HMBKP_EXTENSION_URL', $page );
		}

		if ( ! defined( 'MAINWP_HMBKP_ADMIN_PAGE' ) ) {
			$prefix = 'mainwp_page_';
			$page_slug = 'Extensions-Mainwp-Backupwordpress-Extension';
			if ( \MainWP_BackUp_WordPress::is_managesites_page() ) {
				$page_slug = 'ManageSitesBackUpWordPress';
			}
			define( 'MAINWP_HMBKP_ADMIN_PAGE', $prefix . $page_slug );
		}

		define( 'MAINWP_HMBKP_SECURE_KEY', $this->generate_key() );
	}

	/**
	 * Load all BackUpWordPress functions.
	 */
	protected function includes() {

		require_once( MAINWP_HMBKP_PLUGIN_PATH . 'vendor/autoload.php' );

		require_once( MAINWP_HMBKP_PLUGIN_PATH . 'classes/class-notices.php' );

		// Load the admin menu
		require_once( MAINWP_HMBKP_PLUGIN_PATH . 'admin/menu.php' );
		require_once( MAINWP_HMBKP_PLUGIN_PATH . 'admin/actions.php' );

		// Load Backdrop if necessary.
		if ( ! class_exists( 'HM_Backdrop_Task' ) ) {
			require_once( MAINWP_HMBKP_PLUGIN_PATH . 'backdrop/hm-backdrop.php' );
		}

		require_once( MAINWP_HMBKP_PLUGIN_PATH . 'classes/class-requirements.php' );
		require_once( MAINWP_HMBKP_PLUGIN_PATH . 'classes/class-requirement.php' );

		require_once( MAINWP_HMBKP_PLUGIN_PATH . 'classes/class-path.php' );

		// Load the core backup class
		require_once( MAINWP_HMBKP_PLUGIN_PATH . 'classes/class-backup.php' );

		// Load the backup scheduling classes
		require_once( MAINWP_HMBKP_PLUGIN_PATH . 'classes/class-schedule.php' );
		require_once( MAINWP_HMBKP_PLUGIN_PATH . 'classes/class-schedules.php' );

		// Load the core functions
		require_once( MAINWP_HMBKP_PLUGIN_PATH . 'functions/core.php' );
		require_once( MAINWP_HMBKP_PLUGIN_PATH . 'functions/interface.php' );

		// Load the services
		require_once( MAINWP_HMBKP_PLUGIN_PATH . 'classes/class-services.php' );
		require_once( MAINWP_HMBKP_PLUGIN_PATH . 'classes/class-service.php' );

		// Load the email service
		require_once( MAINWP_HMBKP_PLUGIN_PATH . 'classes/class-email-service.php' );

		require_once( MAINWP_HMBKP_PLUGIN_PATH . 'classes/deprecated.php' );

	}

	/**
	 * Hook into WordPress page lifecycle and execute BackUpWordPress functions.
	 */
	public function hooks() {

		add_action( 'activated_plugin', array( $this, 'load_first' ) );

		add_action( 'admin_init', array( $this, 'upgrade' ) );

		add_action( 'admin_init', array( $this, 'init' ) );

		add_action( 'admin_enqueue_scripts', array( $this, 'scripts' ) );

		add_action( 'admin_footer-' . MAINWP_HMBKP_ADMIN_PAGE, array( $this, 'load_intercom_script' ) );

		add_action( 'admin_enqueue_scripts', array( $this, 'styles' ) );
	}

	/**
	 * Load the Javascript in the admin.
	 *
	 * @param $hook The name of the admin page hook.
	 */
	public function scripts( $hook ) {

		if ( MAINWP_HMBKP_ADMIN_PAGE !== $hook ) {
			return;
		}

		wp_enqueue_script( 'common' );
		wp_enqueue_script( 'wp-lists' );
		wp_enqueue_script( 'postbox' );
		wp_enqueue_script( 'mainwp_backupwp_footer', MAINWP_BACKUPWP_URL . '/js/mainwp-backupwp-footer.js', array( 'jquery' ), false, true );

		//$js_file = MAINWP_HMBKP_PLUGIN_URL . 'assets/hmbkp.min.js';
		// TODO shuold this also support WP_SCRIPT_DEBUG
		//if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
		$js_file = MAINWP_HMBKP_PLUGIN_URL . 'assets/hmbkp.js';
		//}

		wp_enqueue_script( 'mainwp-hmbkp', $js_file, array( 'heartbeat' ), sanitize_key( self::PLUGIN_VERSION ) );

		wp_localize_script(
			'mainwp-hmbkp', 'mainwp_hmbkp_loc', array(
			'page_slug' => MAINWP_HMBKP_PLUGIN_SLUG,
			'nonce' => wp_create_nonce( 'mainwp_hmbkp_nonce' ),
			'mainwp_hmbkp_run_schedule_nonce' => wp_create_nonce( 'mainwp_hmbkp_run_schedule' ),
			'update' => __( 'Update', 'mainwp-backupwordpress-extension' ),
			'cancel' => __( 'Cancel', 'mainwp-backupwordpress-extension' ),
			'delete_schedule' => __( 'Are you sure you want to delete this schedule? All of it\'s backups will also be deleted.', 'mainwp-backupwordpress-extension' ) . "\n\n" . __( '\'Cancel\' to go back, \'OK\' to delete.', 'mainwp-backupwordpress-extension' ) . "\n",
			'delete_backup' => __( 'Are you sure you want to delete this backup?', 'mainwp-backupwordpress-extension' ) . "\n\n" . __( '\'Cancel\' to go back, \'OK\' to delete.', 'mainwp-backupwordpress-extension' ) . "\n",
			'remove_exclude_rule' => __( 'Are you sure you want to remove this exclude rule?', 'mainwp-backupwordpress-extension' ) . "\n\n" . __( '\'Cancel\' to go back, \'OK\' to delete.', 'mainwp-backupwordpress-extension' ) . "\n",
			)
		);
	}

	public function sanitize_key( $key, $raw_key ) {
		if ( 'mainwp_page_Extensions-Mainwp-Backupwordpress-Extension' == $raw_key || 'mainwp_page_Extensions-Mainwp-Ithemes-Security-Extension' == $raw_key ) {
			return $raw_key; }
		return $key;
	}

	/**
	 * Loads the plugin text domain for translation.
	 * This setup allows a user to just drop his custom translation files into the WordPress language directory
	 * Files will need to be in a subdirectory with the name of the textdomain 'mainwp-backupwordpress-extension'
	 */
	public function text_domain() {

		// Set unique textdomain string
		$textdomain = 'mainwp-backupwordpress-extension';

		// The 'plugin_locale' filter is also used by default in load_plugin_textdomain()
		$locale = apply_filters( 'plugin_locale', get_locale(), $textdomain );

		// Set filter for WordPress languages directory
		$mainwp_hmbkp_wp_lang_dir = apply_filters( 'mainwp_hmbkp_do_filter_wp_lang_dir', trailingslashit( WP_LANG_DIR ) . trailingslashit( $textdomain ) . $textdomain . '-' . $locale . '.mo' );

		// Translations: First, look in WordPress' "languages" folder = custom & update-secure!
		load_textdomain( $textdomain, $mainwp_hmbkp_wp_lang_dir );

		// Translations: Secondly, look in plugin's "languages" folder = default
		load_plugin_textdomain( $textdomain, false, MAINWP_HMBKP_PLUGIN_LANG_DIR );
	}

	/**
	 * Determine if we need to run an upgrade routine.
	 */
	public function upgrade() {

	}

	/**
	 * Runs on every admin page load
	 */
	public function init() {

	}

	/**
	 * Generate a unique key.
	 *
	 * @return string
	 */
	protected function generate_key() {

		$key = array( ABSPATH, time() );
		$constants = array( 'AUTH_KEY', 'SECURE_AUTH_KEY', 'LOGGED_IN_KEY', 'NONCE_KEY', 'AUTH_SALT', 'SECURE_AUTH_SALT', 'LOGGED_IN_SALT', 'NONCE_SALT', 'SECRET_KEY' );

		foreach ( $constants as $constant ) {
			if ( defined( $constant ) ) {
				$key[] = constant( $constant );
			}
		}

		shuffle( $key );

		return md5( serialize( $key ) );
	}

	/**
	 * Ensure BackUpWordPress is loaded before add-ons, changes the order of the serialized values in the DB field.
	 */
	public function load_first() {

		$active_plugins = get_option( 'active_plugins' );

		$plugin_path = plugin_basename( __FILE__ );

		$key = array_search( $plugin_path, $active_plugins );

		if ( $key > 0 ) {

			array_splice( $active_plugins, $key, 1 );

			array_unshift( $active_plugins, $plugin_path );

			update_option( 'active_plugins', $active_plugins );
		}
	}

	/**
	 * Enqueue the plugin styles.
	 *
	 * @param $hook
	 */
	public function styles( $hook ) {

		if ( MAINWP_HMBKP_ADMIN_PAGE !== $hook ) {
			return;
		}

		//$css_file = MAINWP_HMBKP_PLUGIN_URL . 'assets/hmbkp.min.css';
		//if ( WP_DEBUG ) {
		$css_file = MAINWP_HMBKP_PLUGIN_URL . 'assets/hmbkp.css';
		//}

		wp_enqueue_style( 'mainwp-backupwordpress-extension', $css_file, false, sanitize_key( self::PLUGIN_VERSION ) );
	}

	/**
	 * Load Intercom and send across user information and server info. Only loaded if the user has opted in.
	 *
	 * @param $hook
	 */
	public function load_intercom_script() {

		if ( ! get_option( 'mainwp_hmbkp_enable_support' ) ) {
			return;
		}

		$info = array();

		foreach ( Requirements::get_requirement_groups() as $group ) {
			foreach ( Requirements::get_requirements( $group ) as $requirement ) {
				$info[ $requirement->name() ] = $requirement->result();
			}
		}

		foreach ( Services::get_services() as $file => $service ) {
			array_merge( $info, call_user_func( array( $service, 'intercom_data' ) ) );
		}

		$current_user = wp_get_current_user();

		$info['user_hash'] = hash_hmac( 'sha256', $current_user->user_email, 'fcUEt7Vi4ym5PXdcr2UNpGdgZTEvxX9NJl8YBTxK' );
		$info['email'] = $current_user->user_email;
		$info['created_at'] = strtotime( $current_user->user_registered );
		$info['app_id'] = '7f1l4qyq';
		$info['name'] = $current_user->display_name;
		$info['widget'] = array( 'activator' => '#intercom' );
		?>

		<script id="IntercomSettingsScriptTag">
		            window.intercomSettings = <?php echo json_encode( $info ); ?>;
		</script>
		<script>!function () {
				function e() {
					var a = c.createElement("script");
					a.type = "text/javascript", a.async = !0, a.src = "https://static.intercomcdn.com/intercom.v1.js";
					var b = c.getElementsByTagName("script")[0];
					b.parentNode.insertBefore(a, b)
				}
				var a = window, b = a.Intercom;
				if ("function" == typeof b)
					b("reattach_activator"), b("update", intercomSettings);
				else {
					var c = document, d = function () {
						d.c(arguments)
					};
					d.q = [], d.c = function (a) {
						d.q.push(a)
					}, a.Intercom = d, a.attachEvent ? a.attachEvent("onload", e) : a.addEventListener("load", e, !1)
				}
		            }();</script>

		<?php
	}
}

if ( is_multisite() && ! is_main_site() ) {
	return;
}
Plugin::get_instance();
